;;; wdired.el --- Rename files editing their names in dired buffers

;; Copyright (C) 2001 Juan Len Lahoz Garca

;; Filename: wdired.el
;; Author: Juan Len Lahoz Garca <juan-leon.lahoz@tecsidel.es>
;; Version: 1.8
;; Keywords: dired, environment, files, renaming

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License as
;; published by the Free Software Foundation; either version 2, or (at
;; your option) any later version.

;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; wdired.el (the "w" is for writable) provides an alternative way of
;; renaming files.
;;
;; Have you ever wished to use C-x r t (string-rectangle), M-%
;; (query-replace), M-c (capitalize-word), etc to change the name of
;; the files in a "dired" buffer? Now you can do this. All the power
;; of emacs commands are available to renaming files!
;; 
;; This package provides a function that makes the filenames of a a
;; dired buffer editable, by changing the buffer mode (witch inhibits
;; all of the commands of dired mode). Here you can edit the names of
;; one or more files and directories, and when you press C-c C-c, the
;; renaming takes effect and you are back to dired mode.
;;
;; This mode won't allow you to edit anything outside of the names of
;; the files in the wdired buffer. The return key, as well as C-j and
;; C-o are "unbinded" (but you can still "paste" newlines or use
;; another tricks to shoot yourself). If, despite these precautions,
;; you change something out of the names, or add or remove lines,
;; unpredictable renames would be done. You are warned ;-).
;;
;; Another things you can do with wdired:
;;
;; - To move files to another directory (by typing their path,
;;   absolute or relative, as a part of the new filename).
;;
;; - To change the target of symbolic links.
;;
;; - To change the permission bits of the filenames (in systems with a
;;   working unix-alike `dired-chmod-program'). See and customize the
;;   variable `wdired-allow-to-change-permissions'. To change a single
;;   char (toggling between its two more usual values) you can press
;;   the space bar over it or left-click the mouse. To set any char to
;;   an specific value (this includes the SUID, SGID and STI bits) you
;;   can use the key labeled as the letter you want. Please note that
;;   permissions of the links cannot be changed in that way, because
;;   the change would affect to their targets, and this would not be
;;   WYSIWYG :-).
;;
;; - To mark files for deletion, by deleting their whole filename.
;;
;; I do not have a URL to hang wdired, but you can use the one below
;; to find the latest version:
;;
;; http://groups.google.com/groups?as_ugroup=gnu.emacs.sources&as_q=wdired

;;; Installation:

;; Add this file (byte-compiling it is recommended) to your load-path.
;; Then add one of these set of lines (or similar ones) to your config:
;;
;; This is the easy way:
;;
;; (require 'wdired)
;; (define-key dired-mode-map "r" 'wdired-change-to-wdired-mode)
;;
;; This is recommended way for faster emacs startup time and lower
;; memory consumption, but remind to add these lines before dired.el
;; gets loaded (i.e., near the beginning of your .emacs file):
;;
;; (autoload 'wdired-change-to-wdired-mode "wdired")
;; (add-hook 'dired-load-hook
;;           '(lambda ()
;;              (define-key dired-mode-map "r" 'wdired-change-to-wdired-mode)
;;              (define-key dired-mode-map
;;                [menu-bar immediate wdired-change-to-wdired-mode]
;;                '("Edit File Names" . wdired-change-to-wdired-mode))))
;;
;;
;; Type "M-x customize-group RET wdired" if you want make changes to
;; the default behavior.

;;; Usage:

;; Then, you can start editing the names of the files by typing "r"
;; (or whatever key you choose, or M-x wdired-change-to-wdired-mode).
;; Use C-c C-c when finished or C-c C-k to abort. You can use also the
;; menu options: in dired mode, "Edit File Names" under "Immediate".
;; While editing the names, a new submenu "WDired" is available at top
;; level. You can customize the behavior of this package from this
;; menu.

;;; Change Log:

;; From 1.7 to 1.8
;;
;; - Now permission (access-control) bits of the files can be changed.
;;   Please see the commentary section and the custom variable
;;   `wdired-allow-to-change-permissions' for details.
;;
;; - Added another possible value for the variable
;;   `wdired-always-move-to-filename-beginning', useful to change
;;   permission bits of several files without the cursor jumping to
;;   filenames when changing lines.

;; From 1.6 to 1.7
;;
;; - Fixed a bug reporting the number of errors in Emacs 21 that was
;;   caused by an obsolete use of `concat'.
;;
;; - Now there is the choice to change the target of the links also.
;;   This option is activated by default. See the variable
;;   `wdired-allow-to-redirect-links' for details.
;;
;; - Code cleanups

;; From 0.1 to 1.6

;; - I've moved the list of changes to another file, because it was
;;   huge. Ask me for it or search older versions in google.

;;; TODO:

;; - Make it to work in XEmacs. Any volunteer?

;;; Code:

(eval-when-compile
  (require 'advice)
  (set (make-local-variable 'byte-compile-dynamic) t))

(eval-and-compile
  (require 'dired)
  (autoload 'dired-do-create-files-regexp "dired-aux")
  (autoload 'dired-call-process "dired-aux"))

(defgroup wdired nil
  "Mode to rename files by editing their names in a dired alike buffer."
  :group 'dired)

(defcustom wdired-use-interactive-rename nil
  "*If non-nil, confirmation is required before changing the filenames
edited in a wdired mode. Confirmation is required also for overwriting
files. If nil, no confirmation is required for change the file names,
and the variable wdired-is-ok-overwrite is used to see if it is ok
to overwrite files without asking."
  :type 'boolean
  :group 'wdired)

(defcustom wdired-is-ok-overwrite nil
  "*If non-nil the renames can overwrite files without asking,
if wdired-use-interactive-rename is non-nil."
  :type 'boolean
  :group 'wdired)

(defcustom wdired-always-move-to-filename-beginning nil
  "*If t the \"up\" and \"down\" movement is done as in dired mode,
that is, always move the point to the beginning of the filename at line.

If `sometimes', only move to the beginning of filename if the point is
before it, and `track-eol' is honored. This behavior is very handy
when editing several filenames.

If nil, \"up\" and \"down\" movement is done as in any other buffer."
  :type '(choice (const :tag "As in any other mode" nil)
		 (const :tag "Smart cursor placement" sometimes)
		 (other :tag "As in dired mode" t))
  :group 'wdired)

(defcustom wdired-advise-functions t
  "*If non-nil some editing commands are advised to work different in
wdired mode. These commands are `upcase-word' `capitalize-word'
`downcase-word' `query-replace' `query-replace-regexp'
`replace-string', and the advice makes them to ignore read-only
regions, so no attempts to modify these regions are done by them, and
so they doesn't end prematurely.

Setting this to nil does not unadvise the functions, if they are
already advised, but new Emacs will not advise them"
  :type 'boolean
  :group 'wdired)

(defcustom wdired-allow-to-redirect-links t
  "*If non-nil, the target of the symbolic links can be changed also.
In systems without symbolic links support, this variable has no effect
at all."
  :type 'boolean
  :group 'wdired)

(defcustom wdired-allow-to-change-permissions nil
  "*If non-nil, the permissions bits of the files can be changed also.
To change a single bit, put the cursor over it and press the space
bar, or left click over it. You can also hit the letter you want to
set: if this value is allowed, the character in the buffer will be
changed. Anyway, the point is advanced one position, so, for example,
you can keep the \"x\" key pressed to give execution permissions to
everybody to that file.

The real change of the permissions is done with the external program
`dired-chmod-program', which must exist."
  :type 'boolean
  :group 'wdired)

(define-key dired-mode-map [menu-bar immediate wdired-change-to-wdired-mode]
  '("Edit File Names" . wdired-change-to-wdired-mode))

(defvar wdired-mode-map nil)
(unless wdired-mode-map
  (setq wdired-mode-map (make-sparse-keymap))
  (define-key wdired-mode-map "\C-x\C-s" 'wdired-finish-edit)
  (define-key wdired-mode-map "\C-c\C-c" 'wdired-finish-edit)
  (define-key wdired-mode-map "\C-c\C-k" 'wdired-abort-changes)
  (define-key wdired-mode-map [return]   'wdired-newline)
  (define-key wdired-mode-map "\C-j"     'wdired-newline)
  (define-key wdired-mode-map "\C-o"     'wdired-newline)
  (define-key wdired-mode-map [up]       'wdired-previous-line)
  (define-key wdired-mode-map "\C-p"     'wdired-previous-line)
  (define-key wdired-mode-map [down]     'wdired-next-line)
  (define-key wdired-mode-map "\C-n"     'wdired-next-line)
  (define-key wdired-mode-map [menu-bar wdired]
    (cons "WDired" (make-sparse-keymap "WDired")))
  (define-key wdired-mode-map [menu-bar wdired wdired-customize]
    '("Options" . wdired-customize))
  (define-key wdired-mode-map [menu-bar wdired dashes]
      '("--"))
  (define-key wdired-mode-map [menu-bar wdired wdired-abort-changes]
    '("Abort Changes" . wdired-abort-changes))
  (define-key wdired-mode-map [menu-bar wdired wdired-finish-edit]
    '("Validate Changes" . wdired-finish-edit)))

(defvar wdired-mode-hooks nil
  "Hooks run when changing to wdired mode.")

(defvar wdired-load-hooks nil
  "Hooks run after loading wdired code.")

;; Local variables (put here to avoid compilation gripes)
(defvar wdired-filenames-ori)
(defvar wdired-links-ori)
(defvar wdired-col-perm) ;; Column where the permission bits start
 ;; Lines in which the permission bits has bee modified
(defvar wdired-mod-perm)

(defvar make-symbolic-link) ;Avoid compilation warning in NTEmacs

(defun wdired-mode ()
  "\\<wdired-mode-map>File Names Editing mode.  Press
\\[wdired-finish-edit] to make the changes to take effect and exit.
To abort the edit, use \\[wdired-abort-changes].

Editing things out of the filenames, or adding or deleting lines is
not allowed, because the rest of the buffer is read-only."
  (interactive)
  (error "This mode can be enabled only by `wdired-change-to-wdired-mode'"))
(put 'wdired-mode 'mode-class 'special)


;;;###autoload
(defun wdired-change-to-wdired-mode ()
  "Change the mode of a dired buffer to another in witch filenames are
editable. In this mode the names of the files can be changed, and
after typing C-c C-c the files and directories in disk are renamed.

Editing things out of the filenames, or adding or deleting lines is
not allowed, because the rest of the buffer is read-only."
  (interactive)
  (set (make-local-variable 'wdired-filenames-ori) (wdired-grab-filenames))
  (use-local-map wdired-mode-map)
  (menu-bar-mode (or menu-bar-mode -1)) ;Force redisplay menu
  (setq buffer-read-only nil)
  (dired-unadvertise default-directory)
  (make-local-hook 'kill-buffer-hook)
  (add-hook 'kill-buffer-hook 'wdired-check-kill-buffer nil t)
  (setq major-mode 'wdired-mode)
  (setq mode-name "Edit filenames")
  (setq revert-buffer-function 'wdired-revert)
  (message "Press C-c C-c when finished")
  (wdired-protect-text (current-buffer))
  (if (and wdired-allow-to-redirect-links
	   (fboundp 'make-symbolic-link))
      (set (make-local-variable 'wdired-links-ori)
	   (wdired-grab-symlinks t)))
  (if wdired-allow-to-change-permissions
      (wdired-make-perms-modifiable))
  (run-hooks wdired-mode-hooks))


;; Protect the buffer so only the filenames can be changed
(defun wdired-protect-text (buff)
  (put-text-property 1 2 'front-sticky t)
  (save-excursion
    (goto-char (point-min))
    (let ((b-protection (point)))
      (while (not (eobp))
        (if (and (dired-move-to-filename nil)
                 (not (string-match "/\\.\\.?$"
                                    (or (dired-get-filename nil t) "/.."))))
            (progn
              (put-text-property b-protection (1- (point)) 'read-only t)
              (setq b-protection (dired-move-to-end-of-filename t))))
        (forward-line))
      (put-text-property b-protection (point-max) 'read-only t)
      (set-buffer-modified-p nil)
      (buffer-disable-undo)    ; Changing properties can be "undo-ed",
      (buffer-enable-undo))))  ; and in this way I kill this possibility.


(defun wdired-grab-filenames()
  "Return a list with the names of the files in a \"dired-like\"
buffer."
  (let ((namelist nil)
	(next-filename nil))
    (save-excursion
      (goto-char (point-min))
      (while (not (eobp))
        (setq namelist (cons (dired-get-filename nil t) namelist))
        (forward-line)))
    namelist))


(defun wdired-change-to-dired-mode ()
  "Change the mode to dired."
  (wdired-unprotect-buffer)
  (use-local-map dired-mode-map)
  (menu-bar-mode (or menu-bar-mode -1)) ;Force redisplay menu
  (setq buffer-read-only t)
  (setq major-mode 'dired-mode)
  (setq mode-name "Dired")
  (dired-advertise)
  (remove-hook 'kill-buffer-hook 'wdired-check-kill-buffer t)
  (setq revert-buffer-function 'dired-revert))


(defun wdired-abort-changes ()
  "Abort changes and return to dired mode."
  (interactive)
  (wdired-change-to-dired-mode)
  (revert-buffer))


(defun wdired-finish-edit ()
  "Finish the edition of the filenames in that buffer, performing the
necessary changes in disk. See also `wdired-change-to-wdired-mode' and
`wdired-mode'"
  (interactive)
  (wdired-change-to-dired-mode)
  (let ((filenames-new (wdired-grab-filenames))
	(overwrite (or wdired-is-ok-overwrite 1))
	(changes nil)
	(files-deleted nil)
	(errors 0)
	filename-ori filename-new tmp-value)
    (if (and wdired-allow-to-redirect-links
	     (boundp 'wdired-links-ori) ; could have been changed
	     (fboundp 'make-symbolic-link))
	(progn
	  (setq tmp-value (wdired-do-symlink-changes))
	  (setq errors (cdr tmp-value))
	  (setq changes (car tmp-value))))
    (if (and wdired-allow-to-change-permissions
	     (boundp 'wdired-col-perm)) ; could have been changed
	(progn
	  (setq tmp-value (wdired-do-perm-changes))
	  (setq errors (+ errors (cdr tmp-value)))
	  (setq changes (or changes (car tmp-value)))))
    (while filenames-new
      (setq filename-ori (car wdired-filenames-ori))
      (setq filename-new (car filenames-new))
      (setq wdired-filenames-ori (cdr wdired-filenames-ori))
      (setq filenames-new (cdr filenames-new))
      (if (not (equal filename-new filename-ori))
          (progn
            (setq changes t)
            (if (not filename-new) ;empty filename!
                (setq files-deleted (cons filename-ori files-deleted))
              (progn
                (setq filename-new (substitute-in-file-name filename-new))
                (if wdired-use-interactive-rename
                    (wdired-search-and-rename filename-ori filename-new)
                  (condition-case err
                      (dired-rename-file filename-ori filename-new
                                         overwrite)
                    (error
		     (setq errors (1+ errors))
		     (dired-log (concat "Rename `" filename-ori "' to `"
					filename-new "' failed:\n%s\n")
				err)))))))))
    (if changes
        (revert-buffer) ;The "revert" is necessary to re-sort the buffer
      (message "(No changes to be performed)"))
    (if files-deleted
        (wdired-flag-for-deletion files-deleted))
    (if (> errors 0)
        (dired-log-summary (format "%d rename actions failed" errors) nil))))

;; Renames a file, searching it in a modified dired buffer, in order
;; to be able to use `dired-do-create-files-regexp' and get its
;; "benefits"
(defun wdired-search-and-rename (filename-ori filename-new)
  (save-excursion
    (goto-char (point-max))
    (forward-line -1)
    (let ((exit-while nil)
	  curr-filename)
      (while (not exit-while)
        (setq curr-filename (dired-get-filename nil t))
        (if (and curr-filename
                 (equal (substitute-in-file-name curr-filename) filename-new))
            (progn
              (setq exit-while t)
              (let ((inhibit-read-only t))
                (dired-move-to-filename)
                (search-forward (dired-get-filename 'no-dir) nil t)
                (replace-match (file-name-nondirectory filename-ori) t t))
              (dired-do-create-files-regexp
               (function dired-rename-file)
               "Move" 1 ".*" filename-new nil t))
          (progn
            (forward-line -1)
            (beginning-of-line)
            (setq exit-while (= 1 (point)))))))))

;; marks a list of files for deletion
(defun wdired-flag-for-deletion (filenames-ori)
  (save-excursion
    (goto-char (point-min))
    (while (not (eobp))
      (if (member (dired-get-filename nil t) filenames-ori)
	    (dired-flag-file-deletion 1)
	(forward-line)))))

(defun wdired-customize ()
  "Customize wdired options."
  (interactive)
  (customize-apropos "wdired" 'groups))

(defun wdired-newline ()
  "Do nothing. It has no sense add newlines in wdired mode."
  (interactive))

(defun wdired-unprotect-buffer ()
  "Remove the properties related to prevent changes in the buffer out
of the filenames."
  (let ((inhibit-read-only t))
    (put-text-property (point-min) (point-max) 'read-only nil)
    (put-text-property (point-min) (point-max) 'local-map nil)
    (put-text-property 1 2 'front-sticky nil)))

(defun wdired-revert (&optional arg noconfirm)
  "Discard changes in the buffer and update the changes in the disk."
  (wdired-change-to-dired-mode)
  (revert-buffer)
  (wdired-change-to-wdired-mode))

(defun wdired-check-kill-buffer ()
  (if (and
       (buffer-modified-p)
       (not (y-or-n-p "Buffer changed. Discard changes and kill buffer? ")))
      (error nil)))

(defun wdired-next-line (arg)
  "Move down lines then position at filename or the current column.
See `wdired-always-move-to-filename-beginning'. Optional prefix ARG
says how many lines to move; default is one line."
  (interactive "p")
  (next-line arg)
  (if (or (eq wdired-always-move-to-filename-beginning t)
	  (and wdired-always-move-to-filename-beginning
	       (< (current-column)
		  (save-excursion (dired-move-to-filename)
				  (current-column)))))
      (dired-move-to-filename)))

(defun wdired-previous-line (arg)
  "Move up lines then position at filename or the current column. See
`wdired-always-move-to-filename-beginning'. Optional prefix ARG says
how many lines to move; default is one line."
  (interactive "p")
  (previous-line arg)
  (if (or (eq wdired-always-move-to-filename-beginning t)
	  (and wdired-always-move-to-filename-beginning
	       (< (current-column)
		  (save-excursion (dired-move-to-filename)
				  (current-column)))))
      (dired-move-to-filename)))

;; The following code deals with changing the target of symbolic
;; links.

(defun wdired-grab-symlinks (&optional set-read-write)
  "Return a list with the links in a \"dired-like\" buffer. If
SET-READ-WRITE, remove read-only properties on the text of the content
of the links"
  (let ((namelist nil)
	(inhibit-read-only set-read-write)
	to-link from-link)
    (save-excursion
      (goto-char (point-min))
      (while (not (eobp))
        (if (looking-at dired-re-sym)
            (progn
	      (setq from-link (dired-get-filename))
              (re-search-forward " -> \\(.*\\)$")
	      (setq to-link (match-string-no-properties 1))
              (setq namelist (cons (cons to-link from-link) namelist))
              (if set-read-write
                  (put-text-property (1- (match-beginning 1))
                                     (match-end 1) 'read-only nil))))
        (forward-line)
	(beginning-of-line)))
    namelist))

;; Perform the changes in the target of the changed links
(defun wdired-do-symlink-changes()
  (let ((wdired-links-new (wdired-grab-symlinks))
	(changes nil)
	(errors 0)
	link-to-ori link-to-new link-from)
    (while wdired-links-new
      (setq link-to-ori (caar wdired-links-ori))
      (setq link-to-new (caar wdired-links-new))
      (setq link-from (cdar wdired-links-ori))
      (setq wdired-links-ori (cdr wdired-links-ori))
      (setq wdired-links-new (cdr wdired-links-new))
      (if (not (equal link-to-new link-to-ori))
          (progn
            (setq changes t)
            (if (equal link-to-new "") ;empty filename!
                (setq link-to-new "/dev/null"))
	    (condition-case err
		(progn 
		  (delete-file link-from)
		  (make-symbolic-link
		   (substitute-in-file-name link-to-new) link-from))
		  (error
		   (setq errors (1+ errors))
		   (dired-log (concat "Link `" link-from "' to `"
				      link-to-new "' failed:\n%s\n")
			      err))))))
    (cons changes errors)))

;; The following code is related to advice some interactive functions
;; to make some editing commands in wdired mode not to fail trying to
;; change read-only text. Notice that some advises advice and unadvise
;; them-self to another functions: search-forward and
;; re-search-forward. This is to keep these functions advised only
;; when is necessary. Since they are built-in commands used heavily in
;; lots of places, to have it permanently advised would cause some
;; performance loss.

;; Check that the chars of the last match do not have read-only
;; properties.
(defun wdired-check-match ()
  (let ((pos-beg (- (match-beginning 0) 1))
	(pos-end (- (match-end 0) 1)))
    (setq pos-beg (max pos-beg 1))
    (while (and (< pos-beg pos-end)
		(not (get-text-property pos-beg 'read-only)))
      (setq pos-beg (1+ pos-beg)))
    (get-text-property pos-beg 'read-only)))


(defun wdired-add-skip-to-writable (command)
  "Add a advice to COMMAND to skip words until the invocation of the
COMMAND in wdired-mode has not any error. This is useful to avoid the
errors due to attempts to modify read-only texts.

Target of links are skipped also, because usually have no sense
play with their capitalization."
  (eval
   `(defadvice ,command (around wdired-skip-to-writable activate)
      ,(format "Make %s to work better with wdired,\n%s."  command
               "skipping read-only words when invoked without argument")
      (interactive "p")
      (if (and (eq major-mode 'wdired-mode)
               (equal (ad-get-arg 0) 1))
          (let ((done nil))
            (while (not done)
              (condition-case err
                  (progn
                    ad-do-it
                    (setq done t))
                (error
                 (setq done (not (forward-word 1)))))))
        ad-do-it))))


(defun wdired-add-skip-in-replace (command)
  "Add a advice to COMMAND to skip matches in wdired mode until while
they have read-only properties. This is useful to avoid \"read-only\"
errors in search and replace commands."
  (eval
    `(defadvice ,command (around wdired-discard-read-only activate)
       ,(format "Make %s to work better with wdired,\n%s."  command
		"skipping read-only matches when invoked without argument")
       ad-do-it 
       (if (eq major-mode 'wdired-mode)
	   (while (and ad-return-value
		       (wdired-check-match))
	     ad-do-it))
       ad-return-value)))


(defun wdired-add-replace-advice (command)
  "Add a advice to COMMAND to skip matches in wdired mode until while
they have read-only properties. This is useful to avoid \"read-only\"
errors in search and replace commands."
  (eval
   `(defadvice ,command (around wdired-grok-read-only activate)
       ,(format "Make %s to work better with wdired,\n%s."  command
		"skipping read-only matches when invoked without argument")
       (if (eq major-mode 'wdired-mode)
           (progn
             (wdired-add-skip-in-replace 'search-forward)
             (wdired-add-skip-in-replace 're-search-forward)
             (unwind-protect 
                 ad-do-it
               (progn
                 (ad-remove-advice 'search-forward
                                   'around 'wdired-discard-read-only)
                 (ad-remove-advice 're-search-forward
                                   'around 'wdired-discard-read-only)
                 (ad-update 'search-forward)
                 (ad-update 're-search-forward))))
         ad-do-it)
       ad-return-value)))


(if wdired-advise-functions
    (progn
      (mapcar 'wdired-add-skip-to-writable
              '(upcase-word capitalize-word downcase-word))
      (mapcar 'wdired-add-replace-advice
              '(query-replace query-replace-regexp replace-string))))


;; The following code deals with changing the access bits (or
;; permissions) of the files.

(defvar wdired-perm-mode-map nil)
(unless wdired-perm-mode-map
  (setq wdired-perm-mode-map (copy-keymap wdired-mode-map))
  (define-key wdired-perm-mode-map " " 'wdired-toggle-bit)
  (define-key wdired-perm-mode-map "r" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "w" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "x" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "-" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "S" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "s" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "T" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "t" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "s" 'wdired-set-bit)
  (define-key wdired-perm-mode-map "l" 'wdired-set-bit)
  (define-key wdired-perm-mode-map [down-mouse-1] 'wdired-mouse-toggle-bit))

;; Put a local-map to the permission bits of the files, and store the
;; original name and permissions as a property
(defun wdired-make-perms-modifiable()
  (let ((inhibit-read-only t)
	filename)
    (set (make-local-variable 'wdired-col-perm) nil)
    (set (make-local-variable 'wdired-mod-perm) nil)
    (save-excursion
      (goto-char (point-min))
      (while (not (eobp))
	(if (and (not (looking-at dired-re-sym))
		 (setq filename (dired-get-filename nil t)))
	    (progn
	      (re-search-forward dired-re-perms)
	      (or wdired-col-perm
		  (setq wdired-col-perm (- (current-column) 9)))
	      (put-text-property (1+ (match-beginning 0)) (match-end 0)
				 'local-map wdired-perm-mode-map)
	      (put-text-property (match-beginning 0) (match-end 0) 'name-ori
				 (cons (match-string-no-properties 0)
				       filename))))
        (forward-line)
	(beginning-of-line)))))

;; Ugly, but any implementation more elegant and legible is welcomed.
(defun wdired-perm-allowed-in-pos (char pos)
  (if (eq char ?-)           t
    (if (= char ?r)            (= (% pos 3) 0)
      (if (= char ?w)            (= (% pos 3) 1)
        (if (= char ?x)            (= (% pos 3) 2)
          (if (= char ?l)            (= pos 5)
            (if (memq char '(?s ?S))   (memq pos '(2 5))
              (if (memq char '(?t ?T))   (= pos 8)
                nil))))))))

(defun wdired-set-bit ()
  "Set a permission bit character."
  (interactive)
  (if (wdired-perm-allowed-in-pos last-command-char
                                  (- (current-column) wdired-col-perm))
      (let ((new-bit (char-to-string last-command-char))
            (inhibit-read-only t))
        (put-text-property 0 1 'local-map wdired-perm-mode-map new-bit)
        (put-text-property 0 1 'read-only t new-bit)
        (insert new-bit)
        (delete-char 1)
        (add-to-list 'wdired-mod-perm (count-lines (point-min) (point))))
    (forward-char 1)))

(defun wdired-toggle-bit()
  "Toggle the permission bit at point."
  (interactive)
  (let ((inhibit-read-only t)
	(new-bit "-"))
    (if (eq (char-after (point)) ?-)
	(setq new-bit	
	      (if (= (% (- (current-column) wdired-col-perm) 3) 0) "r"
		(if (= (% (- (current-column) wdired-col-perm) 3) 1) "w"
		  "x"))))
    (put-text-property 0 1 'local-map wdired-perm-mode-map new-bit)
    (put-text-property 0 1 'read-only t new-bit)
    (insert new-bit)
    (delete-char 1)
    (add-to-list 'wdired-mod-perm (count-lines (point-min) (point)))))

(defun wdired-mouse-toggle-bit (event)
  "Toggle the permission bit that was left clicked."
  (interactive "e")
  (mouse-set-point event)
  (wdired-toggle-bit))

;; Allowed chars for 4000 bit are Ss  in position 3
;; Allowed chars for 2000 bit are Ssl in position 6
;; Allowed chars for 1000 bit are Tt  in position 9
(defun wdired-perms-to-number (perms)
  (let ((vperm (string-to-vector perms))
	(nperm 0777))
    (if (= (elt (string-to-vector perms) 1) ?-) (setq nperm (- nperm 400)))
    (if (= (elt (string-to-vector perms) 2) ?-) (setq nperm (- nperm 200)))
    (let ((p-bit (elt (string-to-vector perms) 3)))
      (if (memq p-bit '(?- ?S)) (setq nperm (- nperm 100)))
      (if (memq p-bit '(?s ?S)) (setq nperm (+ nperm 4000))))
    (if (= (elt (string-to-vector perms) 4) ?-) (setq nperm (- nperm  40)))
    (if (= (elt (string-to-vector perms) 5) ?-) (setq nperm (- nperm  20)))
    (let ((p-bit (elt (string-to-vector perms) 6)))
      (if (memq p-bit '(?- ?S ?l)) (setq nperm (- nperm 10)))
      (if (memq p-bit '(?s ?S ?l)) (setq nperm (+ nperm 2000))))
    (if (= (elt (string-to-vector perms) 7) ?-) (setq nperm (- nperm   4)))
    (if (= (elt (string-to-vector perms) 8) ?-) (setq nperm (- nperm   2)))
    (let ((p-bit (elt (string-to-vector perms) 9)))
      (if (memq p-bit '(?- ?T)) (setq nperm (- nperm 1)))
      (if (memq p-bit '(?t ?T)) (setq nperm (+ nperm 1000))))
    nperm))

;; Perform the changes in the permissions of the files that have
;; changed.
(defun wdired-do-perm-changes ()
  (let ((changes nil)
	(errors 0)
	filename perms-ori)
    (save-excursion
      (while wdired-mod-perm
	(goto-line (car wdired-mod-perm))
	(setq wdired-mod-perm (cdr wdired-mod-perm))
	(re-search-forward dired-re-perms)
	(setq perms-ori (car (get-text-property (match-beginning 0) 'name-ori)))
	(if (not (equal perms-ori (match-string-no-properties 0)))
	    (progn
	      (setq changes t)
	      (setq filename 
		    (cdr (get-text-property (match-beginning 0) 'name-ori)))
	      (if (not (equal 0 (dired-call-process 
				 dired-chmod-program t 
				 (int-to-string 
				  (wdired-perms-to-number
				   (match-string-no-properties 0)))
				 filename)))
                  (progn
                    (setq errors (1+ errors))
                    (dired-log (concat dired-chmod-program " "
                                       (int-to-string 
                                        (wdired-perms-to-number
                                         (match-string-no-properties 0)))
                                       " `" filename "' failed\n\n"))))))))
    (let ((inhibit-read-only t))
      (put-text-property (point-min) (point-max) 'name-ori nil))
    (cons changes errors)))

(provide 'wdired)
(run-hooks wdired-load-hooks)

;;; wdired.el ends here
